# Use current directory as root
$root = Get-Location
$binPath = Join-Path $root "bin"
$exePath = Join-Path $binPath "kubevpn.exe"
$configYaml = Join-Path $root "config.yaml"
$riaConfig = Join-Path $root "ria-config.yaml"

# 1. Extract ZIP file if bin folder doesn't exist
$zipFile = Join-Path $root "kubevpn_v2.4.3_windows_amd64.zip"
if (!(Test-Path $binPath)) {
    Expand-Archive -Path $zipFile -DestinationPath $root -Force
    Write-Host "Extracted kubevpn_v2.4.3_windows_amd64.zip to $root"
}

# 2. Add kubevpn.exe to PATH if not already present
$envPath = [System.Environment]::GetEnvironmentVariable("Path", "User")
if ($envPath -notlike "*$binPath*") {
    [System.Environment]::SetEnvironmentVariable("Path", "$envPath;$binPath", "User")
    Write-Host "Added $binPath to PATH"
}

# 3. Check kubevpn version
$versionOutput = & "$exePath" version 2>&1
if ($versionOutput -match "version") {
    Write-Host "KubeVPN is working: $versionOutput"
}
else {
    Write-Error "Error: kubevpn is not working or version check failed"
    exit 1
}

# 4. Replace $config_path$ in config.yaml
if (Test-Path $configYaml) {
    # Use the raw path without escaping backslashes
    $updatedContent = (Get-Content $configYaml) -replace '\$config_path\$', $riaConfig

    $updatedContent | Set-Content $configYaml
    Write-Host "Updated \$config_path\$ in config.yaml"

    # Overwrite to user kubevpn config folder
    $userProfile = [System.Environment]::GetFolderPath("UserProfile")
    $targetPath = Join-Path $userProfile ".kubevpn\config.yaml"
    Copy-Item -Path $configYaml -Destination $targetPath -Force
    Write-Host "Copied config.yaml to $targetPath"
} else {
    Write-Error "config.yaml not found"
    exit 1
}

# 5. Replace $http_proxy in ria.config
$httpsProxy = [System.Environment]::GetEnvironmentVariable("HTTPS_PROXY", "User")
if ($httpsProxy -and (Test-Path $riaConfig)) {
    Write-Host "HTTPS_PROXY value: $httpsProxy"
    (Get-Content $riaConfig) -replace '\$HTTPS_PROXY', [Regex]::Escape($httpsProxy) | Set-Content $riaConfig
    Write-Host "Successfully updated \$HTTPS_PROXY in ria_config.yaml"
}
else {
    Write-Warning "HTTPS_PROXY environment variable or ria_config.yaml file not found"
}

# 6. Run kubevpn alias command
Write-Host "Executing: kubevpn alias rdi-dev"
& "$exePath" alias rdi-dev
